<?php
$pageTitle = 'API Documentation';
include '../includes/config.php';
include '../includes/header.php';
?>

<h1>API Documentation</h1>

<h2>Overview</h2>
<p>The RC Licensing System provides a comprehensive REST API for license management, verification, and integration with third-party applications.</p>

<h2>API Base URL</h2>
<pre><code><?php echo htmlspecialchars($mirrorUrl); ?>/api/v1</code></pre>

<h2>Authentication</h2>

<h3>API Key Authentication</h3>
<p>All API requests require authentication using an API key:</p>

<pre><code>curl -H "Authorization: Bearer YOUR_API_KEY" \
  <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/status</code></pre>

<h3>Getting Your API Key</h3>
<p>Your API key is available in your account dashboard. Keep it secure and never share it publicly.</p>

<h2>API Endpoints</h2>

<h3>License Status</h3>

<h4>GET /license/status</h4>
<p>Retrieve current license status</p>

<p><strong>Request:</strong></p>
<pre><code>GET <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/status
Authorization: Bearer YOUR_API_KEY</code></pre>

<p><strong>Response:</strong></p>
<pre><code>{
  "status": "active",
  "type": "cPanel",
  "expires": "2025-12-31",
  "cores": 4,
  "workers": 4,
  "ip": "192.168.1.1",
  "hostname": "server.example.com"
}</code></pre>

<h3>License Verification</h3>

<h4>POST /license/verify</h4>
<p>Verify a license key</p>

<p><strong>Request:</strong></p>
<pre><code>POST <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/verify
Authorization: Bearer YOUR_API_KEY
Content-Type: application/json

{
  "license_key": "YOUR_LICENSE_KEY",
  "ip_address": "192.168.1.1"
}</code></pre>

<p><strong>Response:</strong></p>
<pre><code>{
  "valid": true,
  "type": "cPanel",
  "expires": "2025-12-31",
  "message": "License is valid"
}</code></pre>

<h3>License Update</h3>

<h4>POST /license/update</h4>
<p>Update license information</p>

<p><strong>Request:</strong></p>
<pre><code>POST <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/update
Authorization: Bearer YOUR_API_KEY
Content-Type: application/json

{
  "license_key": "YOUR_LICENSE_KEY",
  "hostname": "server.example.com",
  "ip_address": "192.168.1.1"
}</code></pre>

<p><strong>Response:</strong></p>
<pre><code>{
  "success": true,
  "message": "License updated successfully",
  "updated_at": "2025-01-18T12:00:00Z"
}</code></pre>

<h3>License List</h3>

<h4>GET /licenses</h4>
<p>List all licenses for your account</p>

<p><strong>Request:</strong></p>
<pre><code>GET <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/licenses
Authorization: Bearer YOUR_API_KEY</code></pre>

<p><strong>Response:</strong></p>
<pre><code>{
  "licenses": [
    {
      "id": "lic_123",
      "type": "cPanel",
      "status": "active",
      "expires": "2025-12-31",
      "server": "server1.example.com"
    },
    {
      "id": "lic_124",
      "type": "LiteSpeed",
      "status": "active",
      "expires": "2025-06-30",
      "server": "server2.example.com"
    }
  ]
}</code></pre>

<h3>Diagnostics</h3>

<h4>GET /diagnostics</h4>
<p>Get system diagnostics</p>

<p><strong>Request:</strong></p>
<pre><code>GET <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/diagnostics
Authorization: Bearer YOUR_API_KEY</code></pre>

<p><strong>Response:</strong></p>
<pre><code>{
  "os": "CentOS 7.9",
  "kernel": "3.10.0-1160.49.1.el7.x86_64",
  "cpu_cores": 4,
  "memory": "8GB",
  "disk_free": "100GB",
  "uptime": "45 days"
}</code></pre>

<h2>Error Handling</h2>

<h3>Error Response Format</h3>

<pre><code>{
  "error": true,
  "code": "INVALID_LICENSE",
  "message": "The provided license key is invalid",
  "details": {
    "license_key": "Invalid format"
  }
}</code></pre>

<h3>Common Error Codes</h3>

<table>
    <thead>
        <tr>
            <th>Code</th>
            <th>HTTP Status</th>
            <th>Description</th>
        </tr>
    </thead>
    <tbody>
        <tr>
            <td>UNAUTHORIZED</td>
            <td>401</td>
            <td>Invalid or missing API key</td>
        </tr>
        <tr>
            <td>FORBIDDEN</td>
            <td>403</td>
            <td>Access denied</td>
        </tr>
        <tr>
            <td>NOT_FOUND</td>
            <td>404</td>
            <td>Resource not found</td>
        </tr>
        <tr>
            <td>INVALID_REQUEST</td>
            <td>400</td>
            <td>Invalid request parameters</td>
        </tr>
        <tr>
            <td>INVALID_LICENSE</td>
            <td>400</td>
            <td>License validation failed</td>
        </tr>
        <tr>
            <td>SERVER_ERROR</td>
            <td>500</td>
            <td>Internal server error</td>
        </tr>
    </tbody>
</table>

<h2>Rate Limiting</h2>

<h3>Rate Limits</h3>
<ul>
    <li>Free tier: 100 requests per hour</li>
    <li>Standard tier: 1,000 requests per hour</li>
    <li>Enterprise tier: Unlimited</li>
</ul>

<h3>Rate Limit Headers</h3>
<p>Each response includes rate limit information:</p>
<pre><code>X-RateLimit-Limit: 1000
X-RateLimit-Remaining: 999
X-RateLimit-Reset: 1642521600</code></pre>

<h2>Webhooks</h2>

<h3>Webhook Events</h3>
<p>Subscribe to events:</p>
<ul>
    <li>license.activated</li>
    <li>license.expired</li>
    <li>license.updated</li>
    <li>license.verification_failed</li>
</ul>

<h3>Webhook Payload</h3>
<pre><code>{
  "event": "license.activated",
  "timestamp": "2025-01-18T12:00:00Z",
  "data": {
    "license_id": "lic_123",
    "type": "cPanel",
    "server": "server.example.com"
  }
}</code></pre>

<h2>Code Examples</h2>

<h3>PHP Example</h3>
<pre><code>&lt;?php
$api_key = 'YOUR_API_KEY';
$url = '<?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/status';

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $api_key,
    'Content-Type: application/json'
]);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

$response = curl_exec($ch);
$data = json_decode($response, true);

echo $data['status'];
?&gt;</code></pre>

<h3>Python Example</h3>
<pre><code>import requests

api_key = 'YOUR_API_KEY'
url = '<?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/status'

headers = {
    'Authorization': f'Bearer {api_key}',
    'Content-Type': 'application/json'
}

response = requests.get(url, headers=headers)
data = response.json()

print(data['status'])</code></pre>

<h3>cURL Example</h3>
<pre><code>curl -X GET \
  -H "Authorization: Bearer YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  <?php echo htmlspecialchars($mirrorUrl); ?>/api/v1/license/status</code></pre>

<h2>Best Practices</h2>

<ul>
    <li>Always use HTTPS for API requests</li>
    <li>Keep your API key secure</li>
    <li>Implement proper error handling</li>
    <li>Use exponential backoff for retries</li>
    <li>Cache responses when appropriate</li>
    <li>Monitor rate limits</li>
</ul>

<?php include '../includes/footer.php'; ?>
